<?php
/**
 * Background removal media library integration.
 *
 * @package Srizon_AI_Image
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Background removal media library integration class.
 */
class Srizon_AI_Image_BG_Removal {

	/**
	 * Single instance.
	 *
	 * @var Srizon_AI_Image_BG_Removal
	 */
	private static $instance = null;

	/**
	 * Supported image types.
	 *
	 * @var array
	 */
	const SUPPORTED_TYPES = array( 'image/jpeg', 'image/png', 'image/webp' );

	/**
	 * Get single instance.
	 *
	 * @return Srizon_AI_Image_BG_Removal
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		// AJAX handlers.
		add_action( 'wp_ajax_srizon_ai_bg_remove', array( $this, 'ajax_remove_background' ) );
		add_action( 'wp_ajax_srizon_ai_bg_check', array( $this, 'ajax_check_images' ) );

		// Add bulk action for list view.
		add_filter( 'bulk_actions-upload', array( $this, 'add_bulk_action' ) );
		add_filter( 'handle_bulk_actions-upload', array( $this, 'handle_bulk_action' ), 10, 3 );
		add_action( 'admin_notices', array( $this, 'bulk_action_notices' ) );

		// Add action link in attachment details.
		add_filter( 'attachment_fields_to_edit', array( $this, 'add_attachment_field' ), 10, 2 );
	}

	/**
	 * Add bulk action to media library list view.
	 *
	 * @param array $actions Existing bulk actions.
	 * @return array Modified actions.
	 */
	public function add_bulk_action( $actions ) {
		if ( Srizon_AI_Image_Admin::is_api_key_configured() ) {
			$actions['srizon_ai_remove_background'] = __( 'Remove Background', 'srizon-ai-image' );
		}
		return $actions;
	}

	/**
	 * Handle bulk action.
	 *
	 * @param string $redirect_url Redirect URL.
	 * @param string $action       Action name.
	 * @param array  $post_ids     Selected post IDs.
	 * @return string Modified redirect URL.
	 */
	public function handle_bulk_action( $redirect_url, $action, $post_ids ) {
		if ( 'srizon_ai_remove_background' !== $action ) {
			return $redirect_url;
		}

		$processed = 0;
		$errors    = 0;

		foreach ( $post_ids as $post_id ) {
			$result = $this->process_image( $post_id );
			if ( is_wp_error( $result ) ) {
				++$errors;
			} else {
				++$processed;
			}
		}

		$redirect_url = add_query_arg(
			array(
				'srizon_ai_bg_processed' => $processed,
				'srizon_ai_bg_errors'    => $errors,
			),
			$redirect_url
		);

		return $redirect_url;
	}

	/**
	 * Display bulk action notices.
	 */
	public function bulk_action_notices() {
		if ( ! isset( $_GET['srizon_ai_bg_processed'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		$processed = intval( $_GET['srizon_ai_bg_processed'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$errors    = isset( $_GET['srizon_ai_bg_errors'] ) ? intval( $_GET['srizon_ai_bg_errors'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( $processed > 0 ) {
			printf(
				'<div class="notice notice-success is-dismissible"><p>%s</p></div>',
				esc_html(
					sprintf(
						/* translators: %d: Number of processed images */
						_n(
							'Background removed from %d image.',
							'Background removed from %d images.',
							$processed,
							'srizon-ai-image'
						),
						$processed
					)
				)
			);
		}

		if ( $errors > 0 ) {
			printf(
				'<div class="notice notice-error is-dismissible"><p>%s</p></div>',
				esc_html(
					sprintf(
						/* translators: %d: Number of failed images */
						_n(
							'Failed to process %d image.',
							'Failed to process %d images.',
							$errors,
							'srizon-ai-image'
						),
						$errors
					)
				)
			);
		}
	}

	/**
	 * Add field to attachment edit screen.
	 *
	 * @param array   $form_fields Existing form fields.
	 * @param WP_Post $post        Attachment post object.
	 * @return array Modified form fields.
	 */
	public function add_attachment_field( $form_fields, $post ) {
		// Only show for supported image types.
		if ( ! in_array( $post->post_mime_type, self::SUPPORTED_TYPES, true ) ) {
			return $form_fields;
		}

		// Check if this is already a processed image.
		$is_processed = get_post_meta( $post->ID, '_srizon_bg_processed', true );
		if ( $is_processed ) {
			$original_id = get_post_meta( $post->ID, '_srizon_bg_original_id', true );
			$form_fields['srizon_ai_bg_info'] = array(
				'label' => __( 'Background Removal', 'srizon-ai-image' ),
				'input' => 'html',
				'html'  => sprintf(
					'<span class="srizon-ai-bg-processed">%s</span>',
					sprintf(
						/* translators: %d: Original attachment ID */
						esc_html__( 'Processed from original #%d', 'srizon-ai-image' ),
						intval( $original_id )
					)
				),
			);
			return $form_fields;
		}

		// Show remove background button.
		if ( ! Srizon_AI_Image_Admin::is_api_key_configured() ) {
			$form_fields['srizon_ai_bg_remove'] = array(
				'label' => __( 'Background Removal', 'srizon-ai-image' ),
				'input' => 'html',
				'html'  => sprintf(
					'<a href="%s">%s</a>',
					esc_url( admin_url( 'options-general.php?page=srizon-ai-image-settings' ) ),
					esc_html__( 'Configure API key to enable', 'srizon-ai-image' )
				),
			);
		} else {
			$form_fields['srizon_ai_bg_remove'] = array(
				'label' => __( 'Background Removal', 'srizon-ai-image' ),
				'input' => 'html',
				'html'  => sprintf(
					'<button type="button" class="button srizon-ai-bg-remove-single" data-id="%d">%s</button>
					<span class="srizon-ai-bg-status"></span>',
					intval( $post->ID ),
					esc_html__( 'Remove Background', 'srizon-ai-image' )
				),
			);
		}

		return $form_fields;
	}

	/**
	 * AJAX handler to check if images are valid.
	 */
	public function ajax_check_images() {
		check_ajax_referer( 'srizon_ai_bg_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'srizon-ai-image' ) ) );
		}

		$ids = isset( $_POST['ids'] ) ? array_map( 'intval', (array) $_POST['ids'] ) : array();

		if ( empty( $ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No images selected.', 'srizon-ai-image' ) ) );
		}

		$valid   = array();
		$invalid = array();

		foreach ( $ids as $id ) {
			$mime_type = get_post_mime_type( $id );
			if ( in_array( $mime_type, self::SUPPORTED_TYPES, true ) ) {
				// Check if already processed.
				$is_processed = get_post_meta( $id, '_srizon_bg_processed', true );
				if ( ! $is_processed ) {
					$valid[] = $id;
				} else {
					$invalid[] = array(
						'id'     => $id,
						'reason' => __( 'Already processed', 'srizon-ai-image' ),
					);
				}
			} else {
				$invalid[] = array(
					'id'     => $id,
					'reason' => __( 'Unsupported file type', 'srizon-ai-image' ),
				);
			}
		}

		wp_send_json_success(
			array(
				'valid'   => $valid,
				'invalid' => $invalid,
			)
		);
	}

	/**
	 * AJAX handler for background removal.
	 */
	public function ajax_remove_background() {
		check_ajax_referer( 'srizon_ai_bg_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'srizon-ai-image' ) ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( $_POST['attachment_id'] ) : 0;

		if ( ! $attachment_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid attachment ID.', 'srizon-ai-image' ) ) );
		}

		$result = $this->process_image( $attachment_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success(
			array(
				'message'       => __( 'Background removed successfully!', 'srizon-ai-image' ),
				'attachment_id' => $result,
				'url'           => wp_get_attachment_url( $result ),
				'thumbnail'     => wp_get_attachment_image_url( $result, 'thumbnail' ),
			)
		);
	}

	/**
	 * Process a single image for background removal.
	 *
	 * @param int $attachment_id Attachment ID to process.
	 * @return int|WP_Error New attachment ID or error.
	 */
	private function process_image( $attachment_id ) {
		// Validate attachment.
		$attachment = get_post( $attachment_id );
		if ( ! $attachment || 'attachment' !== $attachment->post_type ) {
			return new WP_Error(
				'invalid_attachment',
				__( 'Invalid attachment.', 'srizon-ai-image' )
			);
		}

		// Check mime type.
		if ( ! in_array( $attachment->post_mime_type, self::SUPPORTED_TYPES, true ) ) {
			return new WP_Error(
				'unsupported_type',
				sprintf(
					/* translators: %s: Mime type */
					__( 'Unsupported image type: %s', 'srizon-ai-image' ),
					$attachment->post_mime_type
				)
			);
		}

		// Get image URL.
		$image_url = wp_get_attachment_url( $attachment_id );
		if ( ! $image_url ) {
			return new WP_Error(
				'no_url',
				__( 'Could not get attachment URL.', 'srizon-ai-image' )
			);
		}

		// Call Runware API.
		$api_result = Srizon_AI_Image_API::remove_background( $image_url );

		if ( is_wp_error( $api_result ) ) {
			return $api_result;
		}

		// Save the processed image.
		if ( 'base64' === $api_result['type'] ) {
			$new_attachment_id = Srizon_AI_Image_API::save_bg_removed_base64(
				$api_result['data'],
				$attachment_id
			);
		} else {
			$new_attachment_id = Srizon_AI_Image_API::save_bg_removed_image(
				$api_result['imageURL'],
				$attachment_id
			);
		}

		return $new_attachment_id;
	}
}
