<?php
/**
 * REST API functionality for AI Image generation.
 *
 * @package Srizon_AI_Image
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Srizon_AI_Image_API {

    /**
     * Runware API URL.
     *
     * @var string
     */
    const RUNWARE_API_URL = 'https://api.runware.ai/v1';

    /**
     * Background removal model.
     *
     * @var string
     */
    const BG_REMOVAL_MODEL = 'runware:110@1';

    /**
     * Allowed models for image generation.
     *
     * @var array
     */
    const ALLOWED_MODELS = array(
        'runware:400@1',
        'google:4@1',
        'google:4@2',
        'openai:4@1',
        'midjourney:3@1',
    );

    /**
     * Register REST API routes.
     */
    public function register_routes() {
        register_rest_route(
            'srizon-ai-image/v1',
            '/generate-image',
            array(
                'methods'             => 'POST',
                'callback'            => array( $this, 'generate_image' ),
                'permission_callback' => array( $this, 'check_permissions' ),
                'args'                => array(
                    'prompt' => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ),
                    'width'  => array(
                        'required'          => true,
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                    ),
                    'height' => array(
                        'required'          => true,
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                    ),
                    'model'  => array(
                        'required'          => false,
                        'type'              => 'string',
                        'default'           => 'runware:400@1',
                        'sanitize_callback' => 'sanitize_text_field',
                    ),
                    'steps'  => array(
                        'required'          => false,
                        'type'              => 'integer',
                        'default'           => 0,
                        'sanitize_callback' => 'absint',
                    ),
                    'cfg_scale' => array(
                        'required'          => false,
                        'type'              => 'number',
                        'default'           => 0,
                        'sanitize_callback' => array( $this, 'sanitize_float' ),
                    ),
                    'quality' => array(
                        'required'          => false,
                        'type'              => 'string',
                        'default'           => 'high',
                        'sanitize_callback' => 'sanitize_text_field',
                    ),
                ),
            )
        );
    }

    /**
     * Sanitize float value.
     *
     * @param mixed $value The value to sanitize.
     * @return float
     */
    public function sanitize_float( $value ) {
        return floatval( $value );
    }

    /**
     * Check if user has permission to generate images.
     *
     * @return bool|WP_Error
     */
    public function check_permissions() {
        if ( ! current_user_can( 'upload_files' ) ) {
            return new WP_Error(
                'rest_forbidden',
                __( 'You do not have permission to upload files.', 'srizon-ai-image' ),
                array( 'status' => 403 )
            );
        }

        if ( ! Srizon_AI_Image_Admin::is_api_key_configured() ) {
            return new WP_Error(
                'api_key_missing',
                __( 'Runware API key is not configured.', 'srizon-ai-image' ),
                array( 'status' => 400 )
            );
        }

        return true;
    }

    /**
     * Generate image via Runware API.
     *
     * @param WP_REST_Request $request The request object.
     * @return WP_REST_Response|WP_Error
     */
    public function generate_image( $request ) {
        $prompt    = $request->get_param( 'prompt' );
        $width     = $request->get_param( 'width' );
        $height    = $request->get_param( 'height' );
        $model     = $request->get_param( 'model' );
        $steps     = $request->get_param( 'steps' );
        $cfg_scale = $request->get_param( 'cfg_scale' );
        $quality   = $request->get_param( 'quality' );

        // Validate model.
        if ( ! in_array( $model, self::ALLOWED_MODELS, true ) ) {
            $model = 'runware:400@1';
        }

        $provider = $this->get_provider( $model );

        // Validate quality for OpenAI.
        $allowed_qualities = array( 'auto', 'high', 'medium', 'low' );
        if ( ! in_array( $quality, $allowed_qualities, true ) ) {
            $quality = 'high';
        }

        // Call Runware API.
        $images_data = $this->call_runware_api( $prompt, $width, $height, $model, $steps, $cfg_scale, $quality );

        if ( is_wp_error( $images_data ) ) {
            return new WP_REST_Response(
                array(
                    'success' => false,
                    'message' => $images_data->get_error_message(),
                ),
                400
            );
        }

        // Save all images to media library.
        $attachments = array();
        foreach ( $images_data as $index => $image_data ) {
            $attachment = $this->save_image_to_media_library( $image_data, $prompt, $index + 1, count( $images_data ) );

            if ( is_wp_error( $attachment ) ) {
                // Log error but continue with other images.
                continue;
            }

            $attachments[] = array(
                'attachment_id' => $attachment['id'],
                'url'           => $attachment['url'],
                'alt'           => $prompt,
            );
        }

        if ( empty( $attachments ) ) {
            return new WP_REST_Response(
                array(
                    'success' => false,
                    'message' => __( 'Failed to save images to media library.', 'srizon-ai-image' ),
                ),
                500
            );
        }

        // Return response with all attachments.
        // For backwards compatibility, 'data' contains the first image.
        // 'all_images' contains all images (for Midjourney).
        return new WP_REST_Response(
            array(
                'success'    => true,
                'data'       => $attachments[0],
                'all_images' => $attachments,
                'is_multi'   => count( $attachments ) > 1,
            ),
            200
        );
    }

    /**
     * Get provider name from model string.
     *
     * @param string $model The model identifier.
     * @return string Provider name.
     */
    private function get_provider( $model ) {
        $parts = explode( ':', $model );
        return $parts[0];
    }

    /**
     * Adjust dimensions for provider-specific requirements.
     *
     * @param int    $width    Requested width.
     * @param int    $height   Requested height.
     * @param string $provider Provider name.
     * @return array Adjusted width and height.
     */
    private function adjust_dimensions_for_provider( $width, $height, $provider ) {
        switch ( $provider ) {
            case 'openai':
                // DALL-E 3 only supports: 1024x1024, 1792x1024, 1024x1792
                $ratio = $width / $height;
                if ( abs( $ratio - 1 ) < 0.3 ) {
                    return array( 1024, 1024 );
                } elseif ( $ratio > 1 ) {
                    return array( 1792, 1024 );
                } else {
                    return array( 1024, 1792 );
                }

            case 'midjourney':
                // Midjourney supported dimensions
                $supported = array(
                    array( 1024, 1024 ), // 1:1
                    array( 1456, 816 ),  // 16:9
                    array( 816, 1456 ),  // 9:16
                    array( 1232, 928 ),  // 4:3
                    array( 928, 1232 ),  // 3:4
                    array( 1344, 896 ),  // 3:2
                    array( 896, 1344 ),  // 2:3
                    array( 1680, 720 ),  // 21:9
                );
                return $this->find_closest_dimensions( $width, $height, $supported );

            case 'runware':
            case 'civitai':
                // Dimensions must be divisible by 64
                $width  = (int) ( round( $width / 64 ) * 64 );
                $height = (int) ( round( $height / 64 ) * 64 );
                $width  = max( 512, min( 2048, $width ) );
                $height = max( 512, min( 2048, $height ) );
                return array( $width, $height );

            default:
                return array( $width, $height );
        }
    }

    /**
     * Find closest supported dimensions.
     *
     * @param int   $width     Requested width.
     * @param int   $height    Requested height.
     * @param array $supported Array of supported dimension pairs.
     * @return array Closest supported dimensions.
     */
    private function find_closest_dimensions( $width, $height, $supported ) {
        $target_ratio   = $width / $height;
        $best_match     = $supported[0];
        $best_diff      = PHP_INT_MAX;

        foreach ( $supported as $dims ) {
            $ratio = $dims[0] / $dims[1];
            $diff  = abs( $ratio - $target_ratio );
            if ( $diff < $best_diff ) {
                $best_diff  = $diff;
                $best_match = $dims;
            }
        }

        return $best_match;
    }

    /**
     * Get provider-specific settings.
     *
     * @param string $provider Provider name.
     * @param string $quality  Quality setting for OpenAI models.
     * @return array|null Provider settings or null.
     */
    private function get_provider_settings( $provider, $quality = 'high' ) {
        switch ( $provider ) {
            case 'openai':
                return array(
                    'openai' => array(
                        'quality' => $quality,
                    ),
                );

            case 'google':
                return null;

            case 'midjourney':
                return array(
                    'midjourney' => array(
                        'quality'  => 1,
                        'stylize'  => 100,
                    ),
                );

            default:
                return null;
        }
    }

    /**
     * Get model-specific parameter defaults.
     *
     * @param string $provider Provider name.
     * @param string $model    Model identifier.
     * @return array Parameter defaults with steps and cfg_scale.
     */
    private function get_model_defaults( $provider, $model ) {
        // Model-specific defaults: array( steps_default, steps_min, steps_max, cfg_default, cfg_min, cfg_max, supports_steps, supports_cfg )
        $defaults = array(
            'runware'    => array( 28, 1, 50, 4, 1, 20, true, true ),
            'civitai'    => array( 30, 1, 50, 7, 1, 20, true, true ),
            'google'     => array( 0, 0, 0, 0, 0, 0, false, false ),
            'openai'     => array( 0, 0, 0, 0, 0, 0, false, false ),
            'midjourney' => array( 0, 0, 0, 0, 0, 0, false, false ),
        );

        return isset( $defaults[ $provider ] ) ? $defaults[ $provider ] : array( 30, 1, 50, 7, 1, 20, true, true );
    }

    /**
     * Call Runware API to generate image.
     *
     * @param string $prompt    The image prompt.
     * @param int    $width     Image width.
     * @param int    $height    Image height.
     * @param string $model     The model to use.
     * @param int    $steps     Number of inference steps.
     * @param float  $cfg_scale CFG scale value.
     * @param string $quality   Quality setting for OpenAI models.
     * @return array|WP_Error Image data or error.
     */
    private function call_runware_api( $prompt, $width, $height, $model, $steps = 0, $cfg_scale = 0, $quality = 'high' ) {
        $api_key   = Srizon_AI_Image_Admin::get_api_key();
        $task_uuid = wp_generate_uuid4();
        $provider  = $this->get_provider( $model );

        // Adjust dimensions for provider requirements.
        list( $width, $height ) = $this->adjust_dimensions_for_provider( $width, $height, $provider );

        // Get model defaults.
        $model_defaults = $this->get_model_defaults( $provider, $model );

        // Build request body.
        $task = array(
            'taskType'       => 'imageInference',
            'taskUUID'       => $task_uuid,
            'model'          => $model,
            'positivePrompt' => $prompt,
            'width'          => $width,
            'height'         => $height,
            'numberResults'  => 'midjourney' === $provider ? 4 : 1,
            'outputType'     => 'base64Data',
            'outputFormat'   => 'PNG',
        );

        // Add steps if supported by model.
        if ( $model_defaults[6] ) { // supports_steps
            $steps = $steps > 0 ? $steps : $model_defaults[0];
            $steps = max( $model_defaults[1], min( $model_defaults[2], $steps ) );
            $task['steps'] = $steps;
        }

        // Add CFG scale if supported by model.
        if ( $model_defaults[7] ) { // supports_cfg
            $cfg_scale = $cfg_scale > 0 ? $cfg_scale : $model_defaults[3];
            $cfg_scale = max( $model_defaults[4], min( $model_defaults[5], $cfg_scale ) );
            $task['CFGScale'] = $cfg_scale;
        }

        // Add provider-specific settings.
        $provider_settings = $this->get_provider_settings( $provider, $quality );
        if ( $provider_settings ) {
            $task['providerSettings'] = $provider_settings;
        }

        $request_body = array( $task );

        $response = wp_remote_post(
            self::RUNWARE_API_URL,
            array(
                'headers' => array(
                    'Content-Type'  => 'application/json',
                    'Authorization' => 'Bearer ' . $api_key,
                ),
                'body'    => wp_json_encode( $request_body ),
                'timeout' => 120,
            )
        );

        if ( is_wp_error( $response ) ) {
            return new WP_Error(
                'api_request_failed',
                __( 'Failed to connect to Runware API: ', 'srizon-ai-image' ) . $response->get_error_message()
            );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data          = json_decode( $response_body, true );

        if ( $response_code !== 200 ) {
            $error_message = isset( $data['error'] )
                ? $data['error']
                : ( isset( $data['errors'][0]['message'] ) ? $data['errors'][0]['message'] : __( 'Unknown API error occurred.', 'srizon-ai-image' ) );

            return new WP_Error( 'api_error', $error_message );
        }

        // Extract images from response
        if ( empty( $data['data'] ) || ! is_array( $data['data'] ) ) {
            return new WP_Error(
                'no_image_generated',
                __( 'No image was generated. Please try a different prompt.', 'srizon-ai-image' )
            );
        }

        // Collect all image inference results.
        $images = array();
        foreach ( $data['data'] as $result ) {
            if ( isset( $result['taskType'] ) && 'imageInference' === $result['taskType'] && ! empty( $result['imageBase64Data'] ) ) {
                $images[] = array(
                    'base64' => $result['imageBase64Data'],
                    'mime'   => 'image/png',
                );
            }
        }

        if ( empty( $images ) ) {
            return new WP_Error(
                'no_image_generated',
                __( 'No image was generated. Please try a different prompt.', 'srizon-ai-image' )
            );
        }

        return $images;
    }

    /**
     * Get aspect ratio string from dimensions.
     *
     * @param int $width  Image width.
     * @param int $height Image height.
     * @return string Aspect ratio.
     */
    private function get_aspect_ratio( $width, $height ) {
        $ratio = $width / $height;

        // Map to supported aspect ratios.
        if ( abs( $ratio - 1 ) < 0.1 ) {
            return '1:1';
        } elseif ( abs( $ratio - ( 16 / 9 ) ) < 0.1 ) {
            return '16:9';
        } elseif ( abs( $ratio - ( 9 / 16 ) ) < 0.1 ) {
            return '9:16';
        } elseif ( abs( $ratio - ( 4 / 3 ) ) < 0.1 ) {
            return '4:3';
        } elseif ( abs( $ratio - ( 3 / 4 ) ) < 0.1 ) {
            return '3:4';
        } elseif ( $ratio > 1 ) {
            return '16:9';
        } else {
            return '9:16';
        }
    }

    /**
     * Save generated image to WordPress media library.
     *
     * @param array  $image_data  Image data with base64 and mime type.
     * @param string $prompt      The original prompt for alt text.
     * @param int    $image_num   Image number (for multi-image results like Midjourney).
     * @param int    $total_images Total number of images being saved.
     * @return array|WP_Error Attachment data or error.
     */
    private function save_image_to_media_library( $image_data, $prompt, $image_num = 1, $total_images = 1 ) {
        require_once ABSPATH . 'wp-admin/includes/image.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';

        // Decode base64 image.
        $image_binary = base64_decode( $image_data['base64'] );

        if ( false === $image_binary ) {
            return new WP_Error(
                'decode_failed',
                __( 'Failed to decode image data.', 'srizon-ai-image' )
            );
        }

        // Generate unique filename.
        $filename = 'ai-image-' . wp_generate_uuid4() . '.png';

        // Get WordPress upload directory.
        $upload_dir = wp_upload_dir();

        if ( ! empty( $upload_dir['error'] ) ) {
            return new WP_Error( 'upload_dir_error', $upload_dir['error'] );
        }

        $file_path = $upload_dir['path'] . '/' . $filename;

        // Save file.
        $saved = file_put_contents( $file_path, $image_binary );

        if ( false === $saved ) {
            return new WP_Error(
                'save_failed',
                __( 'Failed to save image file.', 'srizon-ai-image' )
            );
        }

        // Prepare title with image number if multiple images.
        $title = sanitize_file_name( substr( $prompt, 0, 100 ) );
        if ( $total_images > 1 ) {
            $title .= ' (' . $image_num . '/' . $total_images . ')';
        }

        // Prepare attachment data.
        $attachment = array(
            'post_mime_type' => $image_data['mime'],
            'post_title'     => $title,
            'post_content'   => '',
            'post_status'    => 'inherit',
        );

        // Insert attachment.
        $attachment_id = wp_insert_attachment( $attachment, $file_path );

        if ( is_wp_error( $attachment_id ) ) {
            wp_delete_file( $file_path );
            return $attachment_id;
        }

        // Generate attachment metadata.
        $metadata = wp_generate_attachment_metadata( $attachment_id, $file_path );
        wp_update_attachment_metadata( $attachment_id, $metadata );

        // Set alt text.
        update_post_meta( $attachment_id, '_wp_attachment_image_alt', sanitize_text_field( $prompt ) );

        return array(
            'id'  => $attachment_id,
            'url' => wp_get_attachment_url( $attachment_id ),
        );
    }

    /**
     * Remove background from an image.
     *
     * @param string $image_url URL of the image to process.
     * @return array|WP_Error Result with image URL or error.
     */
    public static function remove_background( $image_url ) {
        $api_key = Srizon_AI_Image_Admin::get_api_key();

        if ( empty( $api_key ) ) {
            return new WP_Error(
                'no_api_key',
                __( 'Runware API key is not configured.', 'srizon-ai-image' )
            );
        }

        $task_uuid = wp_generate_uuid4();

        $request_body = array(
            array(
                'taskType'     => 'removeBackground',
                'taskUUID'     => $task_uuid,
                'inputImage'   => $image_url,
                'model'        => self::BG_REMOVAL_MODEL,
                'outputType'   => 'URL',
                'outputFormat' => 'PNG',
                'includeCost'  => true,
            ),
        );

        $response = wp_remote_post(
            self::RUNWARE_API_URL,
            array(
                'timeout' => 60,
                'headers' => array(
                    'Content-Type'  => 'application/json',
                    'Authorization' => 'Bearer ' . $api_key,
                ),
                'body'    => wp_json_encode( $request_body ),
            )
        );

        if ( is_wp_error( $response ) ) {
            return new WP_Error(
                'api_request_failed',
                sprintf(
                    /* translators: %s: Error message */
                    __( 'API request failed: %s', 'srizon-ai-image' ),
                    $response->get_error_message()
                )
            );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data          = json_decode( $response_body, true );

        if ( 200 !== $response_code ) {
            $error_message = isset( $data['error'] ) ? $data['error'] : __( 'Unknown API error', 'srizon-ai-image' );
            if ( isset( $data['message'] ) ) {
                $error_message = $data['message'];
            }
            if ( isset( $data['errors'] ) && is_array( $data['errors'] ) ) {
                $error_messages = array();
                foreach ( $data['errors'] as $error ) {
                    if ( isset( $error['message'] ) ) {
                        $error_messages[] = $error['message'];
                    }
                }
                if ( ! empty( $error_messages ) ) {
                    $error_message = implode( ', ', $error_messages );
                }
            }

            return new WP_Error(
                'api_error',
                sprintf(
                    /* translators: 1: HTTP status code, 2: Error message */
                    __( 'API error (HTTP %1$d): %2$s', 'srizon-ai-image' ),
                    $response_code,
                    $error_message
                )
            );
        }

        // Parse successful response.
        if ( ! isset( $data['data'] ) || ! is_array( $data['data'] ) || empty( $data['data'] ) ) {
            return new WP_Error(
                'invalid_response',
                __( 'Invalid API response format.', 'srizon-ai-image' )
            );
        }

        $result = $data['data'][0];

        if ( ! isset( $result['imageURL'] ) ) {
            // Check for base64 data.
            if ( isset( $result['imageBase64Data'] ) ) {
                return array(
                    'type' => 'base64',
                    'data' => $result['imageBase64Data'],
                    'cost' => isset( $result['cost'] ) ? $result['cost'] : null,
                );
            }

            return new WP_Error(
                'no_image_url',
                __( 'No image URL in API response.', 'srizon-ai-image' )
            );
        }

        return array(
            'type'     => 'url',
            'imageURL' => $result['imageURL'],
            'cost'     => isset( $result['cost'] ) ? $result['cost'] : null,
        );
    }

    /**
     * Download image from URL and save to WordPress media library for background removal.
     *
     * @param string $image_url   URL of the processed image.
     * @param int    $original_id Original attachment ID.
     * @return int|WP_Error New attachment ID or error.
     */
    public static function save_bg_removed_image( $image_url, $original_id ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Get original attachment info.
        $original_file = get_attached_file( $original_id );
        if ( ! $original_file ) {
            return new WP_Error(
                'original_not_found',
                __( 'Original attachment not found.', 'srizon-ai-image' )
            );
        }

        $path_info     = pathinfo( $original_file );
        $original_name = $path_info['filename'];
        $new_filename  = $original_name . '-nobg.png';

        // Download the processed image.
        $temp_file = download_url( $image_url );

        if ( is_wp_error( $temp_file ) ) {
            return new WP_Error(
                'download_failed',
                sprintf(
                    /* translators: %s: Error message */
                    __( 'Failed to download processed image: %s', 'srizon-ai-image' ),
                    $temp_file->get_error_message()
                )
            );
        }

        // Prepare file array for sideloading.
        $file_array = array(
            'name'     => $new_filename,
            'tmp_name' => $temp_file,
        );

        // Get original post parent.
        $original_post = get_post( $original_id );
        $post_parent   = $original_post ? $original_post->post_parent : 0;

        // Sideload the image into the media library.
        $attachment_id = media_handle_sideload( $file_array, $post_parent );

        // Clean up temp file if sideload failed.
        if ( is_wp_error( $attachment_id ) ) {
            wp_delete_file( $temp_file );
            return new WP_Error(
                'sideload_failed',
                sprintf(
                    /* translators: %s: Error message */
                    __( 'Failed to save image to media library: %s', 'srizon-ai-image' ),
                    $attachment_id->get_error_message()
                )
            );
        }

        // Copy relevant metadata from original.
        $original_meta = wp_get_attachment_metadata( $original_id );
        if ( $original_meta && isset( $original_meta['image_meta'] ) ) {
            $new_meta = wp_get_attachment_metadata( $attachment_id );
            if ( $new_meta ) {
                $new_meta['image_meta'] = $original_meta['image_meta'];
                wp_update_attachment_metadata( $attachment_id, $new_meta );
            }
        }

        // Update attachment post with reference to original.
        $original_title = get_the_title( $original_id );
        wp_update_post(
            array(
                'ID'           => $attachment_id,
                'post_title'   => $original_title . ' (Background Removed)',
                'post_excerpt' => sprintf(
                    /* translators: %d: Original attachment ID */
                    __( 'Background removed version of attachment #%d', 'srizon-ai-image' ),
                    $original_id
                ),
            )
        );

        // Store reference to original attachment.
        update_post_meta( $attachment_id, '_srizon_bg_original_id', $original_id );
        update_post_meta( $attachment_id, '_srizon_bg_processed', true );

        return $attachment_id;
    }

    /**
     * Save base64 image to media library for background removal.
     *
     * @param string $base64_data Base64 encoded image data.
     * @param int    $original_id Original attachment ID.
     * @return int|WP_Error New attachment ID or error.
     */
    public static function save_bg_removed_base64( $base64_data, $original_id ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Get original attachment info.
        $original_file = get_attached_file( $original_id );
        if ( ! $original_file ) {
            return new WP_Error(
                'original_not_found',
                __( 'Original attachment not found.', 'srizon-ai-image' )
            );
        }

        $path_info     = pathinfo( $original_file );
        $original_name = $path_info['filename'];
        $new_filename  = $original_name . '-nobg.png';

        // Decode base64 data.
        $image_data = base64_decode( $base64_data );
        if ( false === $image_data ) {
            return new WP_Error(
                'decode_failed',
                __( 'Failed to decode base64 image data.', 'srizon-ai-image' )
            );
        }

        // Create temp file.
        $temp_file = wp_tempnam( $new_filename );
        if ( ! $temp_file ) {
            return new WP_Error(
                'temp_file_failed',
                __( 'Failed to create temporary file.', 'srizon-ai-image' )
            );
        }

        // Write image data to temp file.
        $written = file_put_contents( $temp_file, $image_data );
        if ( false === $written ) {
            wp_delete_file( $temp_file );
            return new WP_Error(
                'write_failed',
                __( 'Failed to write image data to file.', 'srizon-ai-image' )
            );
        }

        // Prepare file array for sideloading.
        $file_array = array(
            'name'     => $new_filename,
            'tmp_name' => $temp_file,
        );

        // Get original post parent.
        $original_post = get_post( $original_id );
        $post_parent   = $original_post ? $original_post->post_parent : 0;

        // Sideload the image into the media library.
        $attachment_id = media_handle_sideload( $file_array, $post_parent );

        // Clean up temp file if sideload failed.
        if ( is_wp_error( $attachment_id ) ) {
            wp_delete_file( $temp_file );
            return $attachment_id;
        }

        // Copy relevant metadata and update post.
        $original_title = get_the_title( $original_id );
        wp_update_post(
            array(
                'ID'           => $attachment_id,
                'post_title'   => $original_title . ' (Background Removed)',
                'post_excerpt' => sprintf(
                    /* translators: %d: Original attachment ID */
                    __( 'Background removed version of attachment #%d', 'srizon-ai-image' ),
                    $original_id
                ),
            )
        );

        // Store reference to original attachment.
        update_post_meta( $attachment_id, '_srizon_bg_original_id', $original_id );
        update_post_meta( $attachment_id, '_srizon_bg_processed', true );

        return $attachment_id;
    }
}
