/**
 * Srizon AI Image - Background Removal Integration
 *
 * @package Srizon_AI_Image
 */

(function($) {
	'use strict';

	var SrizonBgRemoval = {
		/**
		 * Initialize the module.
		 */
		init: function() {
			this.bindEvents();
			this.addToolbarButton();
		},

		/**
		 * Bind event handlers.
		 */
		bindEvents: function() {
			// Handle single removal button clicks
			$(document).on('click', '.srizon-ai-bg-remove-single', this.handleSingleRemoval.bind(this));
		},

		/**
		 * Add button to media library toolbar (grid view).
		 */
		addToolbarButton: function() {
			var self = this;

			// Wait for media library to be ready
			if (typeof wp !== 'undefined' && wp.media && wp.media.view) {
				// Extend the AttachmentsBrowser view
				var AttachmentsBrowser = wp.media.view.AttachmentsBrowser;

				wp.media.view.AttachmentsBrowser = AttachmentsBrowser.extend({
					createToolbar: function() {
						AttachmentsBrowser.prototype.createToolbar.call(this);

						// Add our button to the toolbar
						this.toolbar.set('srizonBgRemoval', new wp.media.view.Button({
							text: srizonAiBgRemoval.buttonText,
							priority: -50,
							click: function() {
								self.handleBulkRemoval();
							}
						}));
					}
				});
			}

			// Also add button for standalone media library page
			this.addStandaloneButton();
		},

		/**
		 * Add button for standalone media library page.
		 */
		addStandaloneButton: function() {
			var self = this;

			// Check if we're on the media library page
			if ($('body').hasClass('upload-php')) {
				// Wait for the page to fully load
				$(window).on('load', function() {
					// Check for grid mode
					if ($('.media-toolbar-mode-select').length) {
						self.addGridModeButton();
					}
				});

				// Also observe for mode changes
				var observer = new MutationObserver(function(mutations) {
					mutations.forEach(function(mutation) {
						if ($('.media-toolbar-mode-select').length && !$('#srizon-ai-bg-removal-btn').length) {
							self.addGridModeButton();
						}
					});
				});

				observer.observe(document.body, {
					childList: true,
					subtree: true
				});
			}
		},

		/**
		 * Add button in grid mode.
		 */
		addGridModeButton: function() {
			var self = this;

			// Find the select mode toolbar
			var $toolbar = $('.media-toolbar-mode-select .media-toolbar-secondary');

			if ($toolbar.length && !$('#srizon-ai-bg-removal-btn').length) {
				var $button = $('<button type="button" id="srizon-ai-bg-removal-btn" class="button media-button">' +
					srizonAiBgRemoval.buttonText + '</button>');

				$button.on('click', function(e) {
					e.preventDefault();
					self.handleBulkRemoval();
				});

				$toolbar.append($button);
			}
		},

		/**
		 * Handle bulk background removal.
		 */
		handleBulkRemoval: function() {
			var self = this;

			// Check if API key is configured
			if (!srizonAiBgRemoval.hasApiKey) {
				this.showNotice(srizonAiBgRemoval.noApiKey, 'error');
				window.location.href = srizonAiBgRemoval.settingsUrl;
				return;
			}

			// Get selected attachments
			var selectedIds = this.getSelectedAttachments();

			if (selectedIds.length === 0) {
				this.showNotice(srizonAiBgRemoval.noSelection, 'warning');
				return;
			}

			// Validate images first
			this.validateImages(selectedIds, function(valid, invalid) {
				if (valid.length === 0) {
					self.showNotice('No valid images to process.', 'error');
					return;
				}

				// Show confirmation
				var message = 'Remove background from ' + valid.length + ' image(s)?';
				if (invalid.length > 0) {
					message += '\n\n' + invalid.length + ' image(s) will be skipped.';
				}

				if (!confirm(message)) {
					return;
				}

				// Process images
				self.processImages(valid);
			});
		},

		/**
		 * Get selected attachment IDs.
		 *
		 * @return {Array} Array of attachment IDs.
		 */
		getSelectedAttachments: function() {
			var ids = [];

			// Grid view - using wp.media
			if (typeof wp !== 'undefined' && wp.media && wp.media.frame) {
				var selection = wp.media.frame.state().get('selection');
				if (selection) {
					selection.each(function(attachment) {
						ids.push(attachment.get('id'));
					});
				}
			}

			// Fallback - check for selected items in DOM
			if (ids.length === 0) {
				$('.attachment.selected, .attachment.details').each(function() {
					var id = $(this).data('id');
					if (id) {
						ids.push(id);
					}
				});
			}

			return ids;
		},

		/**
		 * Validate images via AJAX.
		 *
		 * @param {Array}    ids      Attachment IDs.
		 * @param {Function} callback Callback function.
		 */
		validateImages: function(ids, callback) {
			$.ajax({
				url: srizonAiBgRemoval.ajaxUrl,
				type: 'POST',
				data: {
					action: 'srizon_ai_bg_check',
					nonce: srizonAiBgRemoval.nonce,
					ids: ids
				},
				success: function(response) {
					if (response.success) {
						callback(response.data.valid, response.data.invalid);
					} else {
						callback([], []);
					}
				},
				error: function() {
					callback([], []);
				}
			});
		},

		/**
		 * Process multiple images.
		 *
		 * @param {Array} ids Attachment IDs to process.
		 */
		processImages: function(ids) {
			var self = this;
			var total = ids.length;
			var processed = 0;
			var errors = 0;

			// Show progress
			this.showProgress(0, total);

			// Process images sequentially to avoid API rate limits
			var processNext = function(index) {
				if (index >= ids.length) {
					// All done
					self.hideProgress();
					var message = processed + ' image(s) processed successfully.';
					if (errors > 0) {
						message += ' ' + errors + ' error(s).';
					}
					self.showNotice(message, errors > 0 ? 'warning' : 'success');

					// Refresh the media library
					if (typeof wp !== 'undefined' && wp.media && wp.media.frame) {
						wp.media.frame.content.get().collection.reset();
						wp.media.frame.content.get().collection.more();
					} else {
						// Fallback: reload the page
						location.reload();
					}
					return;
				}

				self.removeBackground(ids[index], function(success) {
					if (success) {
						processed++;
					} else {
						errors++;
					}
					self.showProgress(processed + errors, total);
					processNext(index + 1);
				});
			};

			processNext(0);
		},

		/**
		 * Remove background from single image.
		 *
		 * @param {number}   attachmentId Attachment ID.
		 * @param {Function} callback     Callback function.
		 */
		removeBackground: function(attachmentId, callback) {
			$.ajax({
				url: srizonAiBgRemoval.ajaxUrl,
				type: 'POST',
				data: {
					action: 'srizon_ai_bg_remove',
					nonce: srizonAiBgRemoval.nonce,
					attachment_id: attachmentId
				},
				success: function(response) {
					callback(response.success, response.data);
				},
				error: function() {
					callback(false, null);
				}
			});
		},

		/**
		 * Handle single image removal (from attachment details).
		 *
		 * @param {Event} e Click event.
		 */
		handleSingleRemoval: function(e) {
			e.preventDefault();
			var self = this;
			var $button = $(e.target);
			var attachmentId = $button.data('id');
			var $status = $button.siblings('.srizon-ai-bg-status');

			if (!srizonAiBgRemoval.hasApiKey) {
				window.location.href = srizonAiBgRemoval.settingsUrl;
				return;
			}

			// Disable button and show processing
			$button.prop('disabled', true).text(srizonAiBgRemoval.processing);
			$status.text('').removeClass('success error');

			this.removeBackground(attachmentId, function(success, data) {
				$button.prop('disabled', false).text(srizonAiBgRemoval.buttonText);

				if (success) {
					$status.addClass('success').text(srizonAiBgRemoval.success);
					// Optionally refresh or show the new image
					if (data && data.url) {
						$status.append(' <a href="' + data.url + '" target="_blank">View</a>');
					}
				} else {
					var errorMsg = data && data.message ? data.message : srizonAiBgRemoval.error;
					$status.addClass('error').text(errorMsg);
				}
			});
		},

		/**
		 * Show progress indicator.
		 *
		 * @param {number} current Current count.
		 * @param {number} total   Total count.
		 */
		showProgress: function(current, total) {
			var $progress = $('#srizon-ai-bg-progress');

			if (!$progress.length) {
				$progress = $('<div id="srizon-ai-bg-progress" class="srizon-ai-bg-progress">' +
					'<div class="srizon-ai-bg-progress-bar"><div class="srizon-ai-bg-progress-fill"></div></div>' +
					'<div class="srizon-ai-bg-progress-text"></div>' +
					'</div>');
				$('body').append($progress);
			}

			var percent = Math.round((current / total) * 100);
			$progress.find('.srizon-ai-bg-progress-fill').css('width', percent + '%');
			$progress.find('.srizon-ai-bg-progress-text').text('Removing backgrounds: ' + current + ' / ' + total);
			$progress.show();
		},

		/**
		 * Hide progress indicator.
		 */
		hideProgress: function() {
			$('#srizon-ai-bg-progress').fadeOut(function() {
				$(this).remove();
			});
		},

		/**
		 * Show notice.
		 *
		 * @param {string} message Notice message.
		 * @param {string} type    Notice type (success, error, warning).
		 */
		showNotice: function(message, type) {
			type = type || 'info';

			var $notice = $('<div class="notice notice-' + type + ' is-dismissible srizon-ai-bg-notice">' +
				'<p>' + message + '</p>' +
				'<button type="button" class="notice-dismiss"></button>' +
				'</div>');

			$('.wrap h1').first().after($notice);

			// Auto-dismiss after 5 seconds
			setTimeout(function() {
				$notice.fadeOut(function() {
					$(this).remove();
				});
			}, 5000);

			// Handle dismiss button
			$notice.find('.notice-dismiss').on('click', function() {
				$notice.fadeOut(function() {
					$(this).remove();
				});
			});
		}
	};

	// Initialize when document is ready
	$(document).ready(function() {
		SrizonBgRemoval.init();
	});

})(jQuery);
